// background.js - Service Worker for Chrome Realtime AI Extension

// ⚠️ 配置要求：请替换为你的 Supabase 凭证
// 获取方式：https://supabase.com/dashboard → Settings → API
const SUPABASE_URL = 'https://qcbmusynjrhkxvnosxab.supabase.co';  // 替换为你的 Supabase URL
const SUPABASE_ANON_KEY = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InFjYm11c3luanJoa3h2bm9zeGFiIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTg1NDM2MzksImV4cCI6MjA3NDExOTYzOX0.8FTaqLib0DyFB-xU1vafXjTSsAE4svfrxNSNrA-tYaM';  // 替换为你的 Supabase Anon Key

// API 配置
const API_BASE_URL = 'https://api.realtime-ai.chat';

// ============================================================================
// 状态管理
// ============================================================================
// 注意：Service Worker 可能随时被终止，因此：
// - Supabase 客户端每次使用时通过 initSupabase() 初始化
// - currentSession 存储在 chrome.storage.session (运行时状态)
// - authSession 存储在 chrome.storage.local (持久化)
// - offscreen 相关状态通过 Promise 管理，不依赖全局变量持久性

let supabase = null;  // Supabase 客户端（每次 Service Worker 启动时重新初始化）
let offscreenDocumentPromise = null;  // Offscreen 创建 Promise
let offscreenReadyPromise = null;  // Offscreen 就绪 Promise
let offscreenReadyResolver = null;  // Offscreen 就绪 resolver
let offscreenIsReady = false;  // Offscreen 是否已就绪（Service Worker 重启后会重置）

// 运行时状态管理函数
/**
 * 获取当前翻译会话
 * @returns {Promise<Object|null>} 会话数据
 */
async function getCurrentSession() {
  const result = await chrome.storage.session.get('currentSession');
  return result.currentSession || null;
}

/**
 * 设置当前翻译会话
 * @param {Object|null} session - 会话数据
 */
async function setCurrentSession(session) {
  if (session) {
    await chrome.storage.session.set({ currentSession: session });
  } else {
    await chrome.storage.session.remove('currentSession');
  }
}

// Service Worker 不支持直接加载外部脚本
// 使用 offscreen document 来处理 TRTC 相关操作
async function setupOffscreenDocument() {
  if (offscreenDocumentPromise) {
    return offscreenDocumentPromise;
  }

  // 创建就绪 Promise
  if (!offscreenReadyPromise) {
    offscreenReadyPromise = new Promise((resolve) => {
      offscreenReadyResolver = resolve;
    });
  }

  offscreenDocumentPromise = (async () => {
    // 检查是否已存在 offscreen document
    const existingContexts = await chrome.runtime.getContexts({
      contextTypes: ['OFFSCREEN_DOCUMENT']
    });

    if (existingContexts.length > 0) {
      // 假设已存在的 offscreen 已经就绪
      offscreenIsReady = true;
      if (offscreenReadyResolver) {
        offscreenReadyResolver();
      }
      return;
    }

    // 创建 offscreen document
    await chrome.offscreen.createDocument({
      url: 'offscreen.html',
      reasons: ['USER_MEDIA'],
      justification: 'TRTC audio streaming and processing'
    });

    // 给 offscreen 一点时间加载
    await new Promise(resolve => setTimeout(resolve, 1000));
  })();

  return offscreenDocumentPromise;
}

// 等待 offscreen document 就绪
async function waitForOffscreenReady() {
  // 如果已经就绪，直接返回
  if (offscreenIsReady) {
    return;
  }

  await setupOffscreenDocument();

  // 等待 offscreen 发送就绪消息，最多等待 10 秒
  const timeout = new Promise((resolve) => {
    setTimeout(() => {
      offscreenIsReady = true;
      resolve();
    }, 10000); // 10 秒超时
  });

  await Promise.race([offscreenReadyPromise, timeout]);
}

// 初始化 Supabase (使用 fetch 方式，不依赖 SDK)
// Service Worker 不支持 dynamic import，我们将使用直接的 REST API 调用
async function initSupabase() {
  // 如果已经初始化，直接返回
  if (supabase) {
    return supabase;
  }

  // Supabase 客户端现在通过 REST API 直接调用
  const client = {
    auth: {
      signInWithOtp: async ({ email }) => {
        const response = await fetch(`${SUPABASE_URL}/auth/v1/otp`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'apikey': SUPABASE_ANON_KEY
          },
          body: JSON.stringify({ email })
        });

        if (!response.ok) {
          const error = await response.json().catch(() => ({}));
          throw new Error(error.message || 'Failed to send OTP');
        }

        return { data: await response.json(), error: null };
      },

      verifyOtp: async ({ email, token, type }) => {
        const response = await fetch(`${SUPABASE_URL}/auth/v1/verify`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'apikey': SUPABASE_ANON_KEY
          },
          body: JSON.stringify({ email, token, type })
        });

        if (!response.ok) {
          const error = await response.json().catch(() => ({}));
          throw new Error(error.message || 'Failed to verify OTP');
        }

        const data = await response.json();
        return { data: { session: data }, error: null };
      },

      signOut: async () => {
        // 清除本地存储即可
        return { error: null };
      },

      refreshSession: async (refreshToken) => {
        const response = await fetch(`${SUPABASE_URL}/auth/v1/token?grant_type=refresh_token`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'apikey': SUPABASE_ANON_KEY
          },
          body: JSON.stringify({ refresh_token: refreshToken })
        });

        if (!response.ok) {
          const error = await response.json().catch(() => ({}));
          throw new Error(error.message || 'Failed to refresh session');
        }

        const data = await response.json();
        return { data: { session: data }, error: null };
      }
    }
  };

  supabase = client;
  return client;
}

// 向 offscreen document 发送消息
async function sendToOffscreen(message) {
  await setupOffscreenDocument();

  return new Promise((resolve, reject) => {
    chrome.runtime.sendMessage(message, (response) => {
      if (chrome.runtime.lastError) {
        console.error('[Realtime AI] sendToOffscreen error:', chrome.runtime.lastError);
        reject(chrome.runtime.lastError);
      } else if (response && response.error) {
        console.error('[Realtime AI] Offscreen returned error:', response.error);
        reject(new Error(response.error));
      } else {
        resolve(response);
      }
    });
  });
}

// ============================================================================
// API 调用
// ============================================================================

/**
 * 检查错误是否为 token 相关错误
 * @param {number} status - HTTP 状态码
 * @param {Object} error - 错误对象
 * @returns {boolean} 是否为 token 错误
 */
function isTokenError(status, error) {
  // 401 未授权状态码
  if (status === 401) return true;

  // 检查错误消息中的 token 相关关键词
  const message = error?.message?.toLowerCase() || '';
  const tokenKeywords = ['jwt', 'token', 'expired', 'unauthorized', 'authentication'];

  return tokenKeywords.some(keyword => message.includes(keyword));
}

/**
 * API 调用函数（带自动 token 刷新）
 * @param {string} endpoint - API 端点路径
 * @param {string} method - HTTP 方法 (GET, POST, etc.)
 * @param {Object|null} body - 请求体数据
 * @param {boolean} isRetry - 是否为重试请求（内部使用）
 * @returns {Promise<Object>} API 响应数据
 * @throws {Error} 当请求失败或未登录时抛出错误
 */
async function apiCall(endpoint, method = 'GET', body = null, isRetry = false) {
  // 检查 token 是否即将过期
  if (!isRetry && await isTokenExpiringSoon()) {
    try {
      await refreshSessionToken();
    } catch (error) {
      console.error('[Realtime AI] Proactive refresh failed:', error);
      // 继续尝试使用现有 token
    }
  }

  const token = await getStoredToken();
  if (!token) {
    throw new Error('未登录');
  }

  const options = {
    method,
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    }
  };

  if (body) {
    options.body = JSON.stringify(body);
  }

  const response = await fetch(`${API_BASE_URL}${endpoint}`, options);

  if (!response.ok) {
    const error = await response.json().catch(() => ({ message: '请求失败' }));

    // 如果是 token 相关错误，尝试刷新 token 并重试一次
    if (isTokenError(response.status, error)) {
      // 只重试一次，避免无限循环
      if (!isRetry) {
        try {
          await refreshSessionToken();
          // 递归调用，标记为重试
          return apiCall(endpoint, method, body, true);
        } catch (refreshError) {
          console.error('[Realtime AI] Token refresh failed:', refreshError);
          throw new Error('登录已过期，请重新登录');
        }
      } else {
        // 重试后仍然失败，清除 token
        await clearToken();
        throw new Error('登录已过期，请重新登录');
      }
    }

    throw new Error(error.message || '请求失败');
  }

  return response.json();
}

/**
 * 获取用户配额信息
 * @returns {Promise<Object>} 配额数据
 */
async function getUserQuota() {
  return apiCall('/api/user/quota');
}

/**
 * 启动翻译任务
 * @param {Object} params - 翻译参数
 * @param {string} params.sourceLanguage - 源语言
 * @param {string[]} params.targetLanguages - 目标语言数组
 * @param {number} params.translationMode - 翻译模式 (1=文字翻译, 2=语音同传)
 * @returns {Promise<Object>} 任务会话数据
 */
async function startInterpretation(params) {
  return apiCall('/api/interpreter/start', 'POST', params);
}

/**
 * 停止翻译任务
 * @param {string} taskId - 任务 ID
 * @returns {Promise<Object>} 停止结果
 */
async function stopInterpretation(taskId) {
  return apiCall('/api/interpreter/stop', 'POST', { taskId: taskId });
}


// 存储和获取 session (包括 access_token, refresh_token, expires_at)
async function storeSession(session) {
  await chrome.storage.local.set({
    authSession: {
      access_token: session.access_token,
      refresh_token: session.refresh_token,
      expires_at: session.expires_at,
      user: session.user
    }
  });
}

async function getStoredSession() {
  const result = await chrome.storage.local.get('authSession');
  return result.authSession;
}

async function getStoredToken() {
  const session = await getStoredSession();
  return session?.access_token;
}

async function clearToken() {
  await chrome.storage.local.remove('authSession');
}

/**
 * 确保 content script 已注入到指定标签页
 * @param {number} tabId - 标签页 ID
 * @returns {Promise<boolean>} 是否成功注入或已存在
 */
async function ensureContentScriptInjected(tabId) {
  if (!tabId) return false;

  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ['content.js']
    });
    return true;
  } catch (error) {
    // 已注入时会报错，这是正常的
    console.debug('[Realtime AI] Content script inject attempt:', error.message);
    return false;
  }
}

// 刷新 session token
let refreshPromise = null; // 防止并发刷新

async function refreshSessionToken() {
  // 如果已经有刷新请求在进行中，等待它完成
  if (refreshPromise) {
    return refreshPromise;
  }

  refreshPromise = (async () => {
    try {
      const session = await getStoredSession();

      if (!session || !session.refresh_token) {
        throw new Error('No refresh token');
      }

      const supabase = await initSupabase();
      const { data, error } = await supabase.auth.refreshSession(session.refresh_token);

      if (error) {
        console.error('[Realtime AI] Failed to refresh session:', error);
        throw error;
      }

      // 存储新的 session
      await storeSession(data.session);

      return data.session.access_token;
    } catch (error) {
      console.error('[Realtime AI] Token refresh failed:', error);
      // 刷新失败，清除 session
      await clearToken();
      throw new Error('登录已过期，请重新登录');
    } finally {
      refreshPromise = null;
    }
  })();

  return refreshPromise;
}

// 检查 token 是否即将过期 (5分钟内)
async function isTokenExpiringSoon() {
  const session = await getStoredSession();
  if (!session || !session.expires_at) {
    return true;
  }

  const now = Math.floor(Date.now() / 1000);
  const timeUntilExpiry = session.expires_at - now;

  // 如果在5分钟内过期，返回 true
  return timeUntilExpiry < 300;
}

// ============================================================================
// TRTC 房间管理（通过 offscreen document）
// ============================================================================

/**
 * 加入 TRTC 房间
 * @param {string|number} roomId - 房间 ID
 * @param {string} userId - 用户 ID
 * @param {string} userSig - 用户签名
 * @param {number} sdkAppId - TRTC SDK App ID
 * @returns {Promise<Object>} 加入房间的结果
 */
async function joinTRTCRoom(roomId, userId, userSig, sdkAppId) {
  // 确保 offscreen document 已就绪
  await waitForOffscreenReady();

  return sendToOffscreen({
    type: 'TRTC_JOIN_ROOM',
    data: {
      roomId: parseInt(roomId),
      userId: userId,
      userSig: userSig,
      sdkAppId: sdkAppId || 1400000000  // 使用 API 返回的 sdkAppId，或默认值
    }
  });
}

/**
 * 离开 TRTC 房间
 * @returns {Promise<Object>} 离开房间的结果
 */
async function leaveTRTCRoom() {
  return sendToOffscreen({
    type: 'TRTC_LEAVE_ROOM'
  });
}


/**
 * 停止音频发布
 * @returns {Promise<Object>} 停止结果
 */
async function unpublishAudio() {
  return sendToOffscreen({
    type: 'TRTC_UNPUBLISH_AUDIO'
  });
}

// 消息监听
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  (async () => {
    try {
      switch (message.type) {
        case 'LOGIN': {
          supabase = await initSupabase();
          const { data, error } = await supabase.auth.signInWithOtp({
            email: message.email
          });
          if (error) throw error;
          sendResponse({ success: true, data });
          break;
        }

        case 'VERIFY_OTP': {
          supabase = await initSupabase();
          const { data: sessionData, error: verifyError } = await supabase.auth.verifyOtp({
            email: message.email,
            token: message.otp,
            type: 'email'
          });
          if (verifyError) throw verifyError;

          // 存储完整 session (包括 refresh_token)
          await storeSession(sessionData.session);
          sendResponse({ success: true, session: sessionData.session });
          break;
        }

        case 'LOGOUT': {
          supabase = await initSupabase();
          await supabase.auth.signOut();
          await clearToken();
          sendResponse({ success: true });
          break;
        }

        case 'GET_QUOTA': {
          const quota = await getUserQuota();
          sendResponse({ success: true, data: quota });
          break;
        }

        case 'START_INTERPRETATION': {
          // 启动翻译任务
          // 将 mode 转换为数字：text = 1, voice = 2
          const translationMode = message.params.mode === 'voice' ? 2 : 1;

          // 确保 content script 已注入到当前标签页
          const [currentTab] = await chrome.tabs.query({ active: true, currentWindow: true });
          if (currentTab) {
            await ensureContentScriptInjected(currentTab.id);
          }

          // 语言映射：优化语音识别参数
          const languageMap = {
            'zh': '16k_zh_en_caption',  // 中文（支持中英混合）
            'en': '16k_en_caption'       // 英文
          };

          const startParams = {
            sourceLanguage: languageMap[message.params.sourceLang] || message.params.sourceLang,
            targetLanguages: [message.params.targetLang],  // 后端需要数组
            translationMode: translationMode  // 1 = 文字翻译, 2 = 语音同传
          };

          // Add voiceId if provided (语音模式下使用)
          if (message.params.voiceId) {
            startParams.voiceId = message.params.voiceId;
          }

          const startResult = await startInterpretation(startParams);

          // 处理实际的 API 响应结构
          const apiResponse = startResult.data || startResult;

          // 验证必需字段
          if (!apiResponse || !apiResponse.userInfo || !apiResponse.userInfo.roomId) {
            console.error('[Realtime AI] Invalid API response:', apiResponse);
            throw new Error('API response missing required fields (userInfo.roomId)');
          }

          // 转换为内部使用的格式并存储到 session storage
          const sessionData = {
            task_id: apiResponse.taskId,
            room_id: apiResponse.userInfo.roomId,
            user_id: apiResponse.userInfo.userId,
            user_sig: apiResponse.userInfo.userSig,
            sdkAppId: apiResponse.userInfo.sdkAppId,
            robot_id: apiResponse.userInfo.robotId,
            mode: message.params.mode,
            tabId: currentTab.id  // 保存标签页 ID，用于跨标签页停止和字幕控制
          };
          await setCurrentSession(sessionData);

          // 加入 TRTC 房间
          await joinTRTCRoom(
            sessionData.room_id,
            sessionData.user_id,
            sessionData.user_sig,
            sessionData.sdkAppId
          );

          /**
           * 音频捕获流程：
           * 1. 使用 chrome.tabCapture.getMediaStreamId 获取 streamId
           * 2. 将 streamId 发送到 offscreen.js
           * 3. offscreen.js 使用 streamId 创建 MediaStream 并发布到 TRTC
           * 
           * 注意：文本模式和语音模式都需要捕获音频
           * - TRTC 需要音频流才能进行实时翻译
           * - 两种模式的区别在于后续处理，不在捕获阶段
           */
          try {
            const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
            if (!tabs[0]) {
              throw new Error('无法获取当前标签页');
            }

            const streamId = await chrome.tabCapture.getMediaStreamId({
              targetTabId: tabs[0].id
            });

            await sendToOffscreen({
              type: 'START_TAB_CAPTURE',
              streamId: streamId
            });
          } catch (error) {
            console.error('[Realtime AI] Tab capture error:', error);

            // 清理已创建的资源
            await leaveTRTCRoom();
            await setCurrentSession(null);

            // 向用户显示错误
            throw new Error('无法捕获标签页音频，请确保授予了音频捕获权限');
          }

          sendResponse({ success: true, session: sessionData });
          break;
        }

        case 'STOP_INTERPRETATION': {
          const currentSession = await getCurrentSession();
          if (currentSession) {
            await stopInterpretation(currentSession.task_id);

            /**
             * 资源清理（所有模式都需要）：
             * - 停止标签页音频捕获
             * - 取消发布 TRTC 音频流
             * - 离开 TRTC 房间
             * - 清除 session 存储
             * - 通知 content script 隐藏字幕
             */
            try {
              await sendToOffscreen({
                type: 'STOP_TAB_CAPTURE'
              });
            } catch (error) {
              console.error('[Realtime AI] Failed to stop tab capture:', error);
            }

            await unpublishAudio();

            await leaveTRTCRoom();
            await setCurrentSession(null);

            // 通知 content script 停止显示字幕（使用保存的 tabId）
            if (currentSession.tabId) {
              try {
                await chrome.tabs.sendMessage(currentSession.tabId, { type: 'STOP_SUBTITLE' });
              } catch (error) {
                console.debug('[Realtime AI] Failed to notify content script to stop subtitle:', error.message);
              }
            }
          }
          sendResponse({ success: true });
          break;
        }

        case 'GET_SESSION': {
          const session = await getCurrentSession();
          sendResponse({ success: true, session: session });
          break;
        }

        case 'TTS_SYNTHESIZE': {
          // 代理 TTS 请求
          (async () => {
            try {
              const { text, voiceId } = message.params;
              const { authSession } = await chrome.storage.local.get('authSession');

              if (!authSession || !authSession.access_token) {
                throw new Error('未登录');
              }

              const response = await fetch(`${API_BASE_URL}/api/tts/synthesize`, {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                  'Authorization': `Bearer ${authSession.access_token}`
                },
                body: JSON.stringify({ text, voiceId })
              });

              if (!response.ok) {
                const errorText = await response.text();
                throw new Error(`TTS API Error: ${response.status} - ${errorText}`);
              }

              const data = await response.json();
              sendResponse({ success: true, data: data });
            } catch (error) {
              console.error('[Realtime AI] TTS error:', error);
              sendResponse({ success: false, error: error.message });
            }
          })();
          return true; // 保持通道开启
        }

        case 'OFFSCREEN_READY': {
          // Offscreen document 已就绪
          offscreenIsReady = true;
          if (offscreenReadyResolver) {
            offscreenReadyResolver();
          }
          sendResponse({ success: true });
          break;
        }

        case 'TRTC_TRANSLATION_MESSAGE': {
          // 翻译消息 (type=10000)，转发到 content script 显示字幕

          const activeTabs = await chrome.tabs.query({ active: true, currentWindow: true });
          if (activeTabs[0]) {
            try {
              await chrome.tabs.sendMessage(activeTabs[0].id, {
                type: 'SHOW_SUBTITLE',
                data: message.data
              });
            } catch (error) {
              console.error('[Realtime AI] 发送字幕消息失败 (content script可能未加载):', error.message);

              // 尝试重新注入 content script
              await ensureContentScriptInjected(activeTabs[0].id);

              // 重试发送消息
              try {
                await chrome.tabs.sendMessage(activeTabs[0].id, {
                  type: 'SHOW_SUBTITLE',
                  data: message.data
                });
              } catch (retryError) {
                console.error('[Realtime AI] 重试发送字幕消息失败:', retryError.message);
              }
            }
          }

          sendResponse({ success: true });
          break;
        }

        default: {
          sendResponse({ success: false, error: '未知消息类型' });
        }
      }
    } catch (error) {
      console.error('[Realtime AI] Background error:', error);
      sendResponse({ success: false, error: error.message });
    }
  })();

  return true; // 保持消息通道开放以支持异步响应
});


// 定时检查并刷新 token (每 4 分钟检查一次)
// Service Worker 可能会被终止，使用 chrome.alarms 更可靠
chrome.alarms.create('refreshToken', { periodInMinutes: 4 });

chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name === 'refreshToken') {
    try {
      const session = await getStoredSession();
      if (session && session.expires_at) {
        const now = Math.floor(Date.now() / 1000);
        const timeUntilExpiry = session.expires_at - now;

        // 如果在 5 分钟内过期，主动刷新
        if (timeUntilExpiry < 300) {
          await refreshSessionToken();
        }
      }
    } catch (error) {
      console.error('[Realtime AI] Alarm refresh error:', error);
    }
  }
});

// ============================================================================
// 全局错误处理
// ============================================================================
self.addEventListener('error', (event) => {
  console.error('[Realtime AI] Global error:', event.error);
});

self.addEventListener('unhandledrejection', (event) => {
  console.error('[Realtime AI] Unhandled promise rejection:', event.reason);
});

// ============================================================================
// Service Worker 启动初始化
// ============================================================================
(async () => {
  try {
    const session = await getStoredSession();
    if (session && session.expires_at) {
      const now = Math.floor(Date.now() / 1000);
      const timeUntilExpiry = session.expires_at - now;

      // 如果已经过期或即将过期，立即刷新
      if (timeUntilExpiry < 300) {
        await refreshSessionToken();
      }
    }
  } catch (error) {
    console.error('[Realtime AI] Startup refresh error:', error);
  }
})();
