// content.js - Content script for Chrome Realtime AI Extension

// 使用全局状态对象避免重复注入时的冲突
window.__realtimeAIScriptState__ = window.__realtimeAIScriptState__ || {
  subtitleContainer: null,
  currentSubtitle: null,
  hideTimeout: null,
  isDragging: false,
  dragOffset: { x: 0, y: 0 },
  currentRoundId: null,   // 当前轮次ID
  currentRoundData: null  // 当前轮次数据
};

const scriptState = window.__realtimeAIScriptState__;

// 创建字幕容器 - 可拖动的透明浮窗
function createSubtitleContainer() {
  if (scriptState.subtitleContainer) {
    return scriptState.subtitleContainer;
  }

  scriptState.subtitleContainer = document.createElement('div');
  scriptState.subtitleContainer.id = 'realtime-ai-subtitle';
  scriptState.subtitleContainer.style.cssText = `
    position: fixed;
    bottom: 100px;
    left: 50%;
    transform: translateX(-50%);
    max-width: 900px;
    width: 85%;
    background: rgba(15, 23, 42, 0.85);
    color: #f8fafc;
    padding: 24px 32px;
    border-radius: 20px;
    font-size: 20px;
    line-height: 1.8;
    z-index: 999999;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04), 0 0 0 1px rgba(148, 163, 184, 0.1);
    backdrop-filter: blur(16px);
    transition: opacity 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    pointer-events: auto;
    display: none;
    text-align: left;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    cursor: move;
    user-select: none;
    border: 1px solid rgba(255, 255, 255, 0.1);
  `;

  // 添加拖动功能
  scriptState.subtitleContainer.addEventListener('mousedown', startDrag);
  document.addEventListener('mousemove', drag);
  document.addEventListener('mouseup', stopDrag);

  document.body.appendChild(scriptState.subtitleContainer);
  return scriptState.subtitleContainer;
}

// 开始拖动
function startDrag(e) {
  scriptState.isDragging = true;
  const rect = scriptState.subtitleContainer.getBoundingClientRect();
  scriptState.dragOffset.x = e.clientX - rect.left;
  scriptState.dragOffset.y = e.clientY - rect.top;
  scriptState.subtitleContainer.style.transition = 'opacity 0.3s';
}

// 拖动中
function drag(e) {
  if (!scriptState.isDragging || !scriptState.subtitleContainer) return;

  e.preventDefault();
  const x = e.clientX - scriptState.dragOffset.x;
  const y = e.clientY - scriptState.dragOffset.y;

  scriptState.subtitleContainer.style.left = `${x}px`;
  scriptState.subtitleContainer.style.top = `${y}px`;
  scriptState.subtitleContainer.style.bottom = 'auto';
  scriptState.subtitleContainer.style.transform = 'none';
}

// 停止拖动
function stopDrag() {
  scriptState.isDragging = false;
}

// 显示字幕（只显示当前一轮，新轮次才替换）
function showSubtitle(roundId, text, translation) {
  const container = createSubtitleContainer();

  // 如果是新的 round_id，清空之前的数据
  if (scriptState.currentRoundId !== roundId) {
    console.log('[Realtime AI] 新的轮次:', roundId, '(上一轮:', scriptState.currentRoundId, ')');
    scriptState.currentRoundId = roundId;
    // 新轮次，重置数据
    scriptState.currentRoundData = { text: '', translation: '' };
  }

  // 累积更新当前轮次数据（只更新非空字段）
  if (text) {
    scriptState.currentRoundData.text = text;
  }
  if (translation) {
    scriptState.currentRoundData.translation = translation;
  }

  // 确保容器已显示
  if (container.style.display === 'none') {
    container.style.display = 'block';
    container.style.opacity = '1';
  }

  // 获取或创建子元素
  let originalLine = container.querySelector('.subtitle-original');
  let translationLine = container.querySelector('.subtitle-translation');

  if (!originalLine) {
    // 初次创建
    originalLine = document.createElement('div');
    originalLine.className = 'subtitle-original';
    originalLine.style.cssText = 'margin-bottom: 8px; font-weight: 400; opacity: 0.85; font-size: 16px; color: #e0e0e0; min-height: 20px; transition: opacity 0.15s ease;';
    container.appendChild(originalLine);
  }

  if (!translationLine) {
    translationLine = document.createElement('div');
    translationLine.className = 'subtitle-translation';
    translationLine.style.cssText = 'color: #fff; font-size: 20px; font-weight: 500; text-shadow: 0 2px 8px rgba(0, 0, 0, 0.3); min-height: 24px; transition: opacity 0.15s ease;';
    container.appendChild(translationLine);
  }

  // 更新文本内容（使用保存的累积数据）
  originalLine.textContent = scriptState.currentRoundData.text || '\u00A0';
  translationLine.textContent = scriptState.currentRoundData.translation || '\u00A0';

  // 不自动隐藏，只有新轮次来的时候才替换
  // 清除之前的隐藏定时器（如果有的话）
  if (scriptState.hideTimeout) {
    clearTimeout(scriptState.hideTimeout);
    scriptState.hideTimeout = null;
  }
}

// 隐藏字幕
function hideSubtitle() {
  if (scriptState.subtitleContainer) {
    scriptState.subtitleContainer.style.opacity = '0';
    setTimeout(() => {
      scriptState.subtitleContainer.style.display = 'none';
    }, 300);
  }
}

// 处理翻译字幕 (TRTC type=10000 消息)
function handleSubtitle(payload) {
  console.log('[Realtime AI] 收到字幕数据:', JSON.stringify(payload, null, 2));

  // 根据 TRTC 消息格式：
  // - roundid: 轮次ID (注意是 roundid 不是 round_id)
  // - text: 原文（识别出的文本）
  // - translation_text: 译文
  // - end: 是否是完整的一句话
  const roundId = payload.roundid || payload.round_id || payload.id || Date.now();
  const originalText = payload.text || '';
  const translationText = payload.translation_text || '';
  const isComplete = payload.end || false;

  console.log('[Realtime AI] 字幕内容:', {
    轮次ID: roundId,
    原文: originalText,
    译文: translationText,
    是否完整: isComplete,
    所有字段: Object.keys(payload)
  });

  // 显示字幕 (原文和翻译)
  // 确保至少有原文或译文之一
  if (originalText || translationText) {
    showSubtitle(roundId, originalText, translationText);
  } else {
    console.warn('[Realtime AI] 字幕数据为空，跳过显示');
  }
}

// 停止字幕显示和清理状态
function stopSubtitleDisplay() {
  hideSubtitle();

  // 清空当前轮次数据
  scriptState.currentRoundId = null;
  scriptState.currentRoundData = null;

  console.log('[Realtime AI] Subtitle display stopped');
}

// 显示通知
function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  notification.style.cssText = `
    position: fixed;
    top: 24px;
    right: 24px;
    max-width: 320px;
    padding: 16px 24px;
    background: ${type === 'error' ? 'rgba(239, 68, 68, 0.9)' : 'rgba(99, 102, 241, 0.9)'};
    color: white;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 500;
    z-index: 1000000;
    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    backdrop-filter: blur(8px);
    border: 1px solid rgba(255, 255, 255, 0.1);
    animation: slideIn 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
  `;

  notification.textContent = message;
  document.body.appendChild(notification);

  setTimeout(() => {
    notification.style.animation = 'slideOut 0.3s ease';
    setTimeout(() => {
      notification.remove();
    }, 300);
  }, 3000);
}

// 提取页面主要文本内容
function extractPageText() {
  console.log('[Realtime AI] 开始提取页面文本');

  // 优先选择器 - 尝试提取主要内容区域
  const contentSelectors = [
    'article',
    'main',
    '[role="main"]',
    '.post-content',
    '.article-content',
    '.content',
    '.entry-content',
    '#content',
    'body'
  ];

  let contentElement = null;

  // 尝试找到最合适的内容容器
  for (const selector of contentSelectors) {
    contentElement = document.querySelector(selector);
    if (contentElement) {
      console.log('[Realtime AI] 找到内容容器:', selector);
      break;
    }
  }

  if (!contentElement) {
    contentElement = document.body;
  }

  // 提取文本 - 跳过脚本、样式、导航等
  const excludeSelectors = 'script, style, nav, header, footer, aside, .advertisement, .ad, [role="navigation"]';
  const clone = contentElement.cloneNode(true);

  // 移除不需要的元素
  clone.querySelectorAll(excludeSelectors).forEach(el => el.remove());

  // 获取纯文本
  let text = clone.innerText || clone.textContent || '';

  // 清理文本 - 移除多余空白和换行
  text = text
    .replace(/\s+/g, ' ')  // 多个空白符替换为单个空格
    .replace(/\n\s*\n/g, '\n')  // 多个换行替换为单个换行
    .trim();

  console.log('[Realtime AI] 提取文本长度:', text.length);

  return text;
}

// 监听来自 popup 和 background 的消息
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'SHOW_SUBTITLE':
      // 显示翻译字幕
      handleSubtitle(message.data);
      sendResponse({ success: true });
      break;

    case 'STOP_SUBTITLE':
      // 停止显示字幕
      stopSubtitleDisplay();
      sendResponse({ success: true });
      break;

    case 'EXTRACT_TEXT':
      // 提取页面文本
      try {
        const text = extractPageText();
        sendResponse({ success: true, text: text });
      } catch (error) {
        console.error('[Realtime AI] 文本提取失败:', error);
        sendResponse({ success: false, error: error.message });
      }
      break;

    default:
      sendResponse({ success: false, error: '未知消息类型' });
  }

  return true;
});

// 添加样式动画
const style = document.createElement('style');
style.textContent = `
  @keyframes slideIn {
    from {
      transform: translateX(400px);
      opacity: 0;
    }
    to {
      transform: translateX(0);
      opacity: 1;
    }
  }

  @keyframes slideOut {
    from {
      transform: translateX(0);
      opacity: 1;
    }
    to {
      transform: translateX(400px);
      opacity: 0;
    }
  }
`;
document.head.appendChild(style);

// ============================================================================
// 侧边栏阅读器功能
// ============================================================================

// 扩展全局状态
Object.assign(scriptState, {
  sidebarContainer: null,
  readerState: {
    isPlaying: false,
    isPaused: false,
    sentences: [],
    currentIndex: 0,
    audioContext: null,
    voiceId: null
  }
});

// 创建侧边栏
function createSidebar() {
  if (scriptState.sidebarContainer) {
    return scriptState.sidebarContainer;
  }

  const container = document.createElement('div');
  container.id = 'realtime-ai-sidebar';
  container.style.cssText = `
    position: fixed;
    top: 0;
    right: 0;
    width: 360px;
    height: 100vh;
    background: rgba(15, 23, 42, 0.95);
    color: #f8fafc;
    z-index: 2147483647;
    box-shadow: -4px 0 20px rgba(0, 0, 0, 0.2);
    backdrop-filter: blur(16px);
    display: flex;
    flex-direction: column;
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    transform: translateX(100%);
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    border-left: 1px solid rgba(255, 255, 255, 0.1);
  `;

  // 头部
  const header = document.createElement('div');
  header.style.cssText = `
    padding: 20px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: rgba(15, 23, 42, 0.5);
  `;

  const title = document.createElement('h2');
  title.textContent = 'AI 阅读模式';
  title.style.cssText = 'margin: 0; font-size: 18px; font-weight: 600;';

  const closeBtn = document.createElement('button');
  closeBtn.innerHTML = '×';
  closeBtn.style.cssText = `
    background: none;
    border: none;
    color: #94a3b8;
    font-size: 24px;
    cursor: pointer;
    padding: 4px 8px;
    border-radius: 4px;
    transition: all 0.2s;
  `;
  closeBtn.onmouseover = () => closeBtn.style.color = '#fff';
  closeBtn.onmouseout = () => closeBtn.style.color = '#94a3b8';
  closeBtn.onclick = stopSidebarReader;

  header.appendChild(title);
  header.appendChild(closeBtn);

  // 内容区域
  const content = document.createElement('div');
  content.id = 'realtime-ai-sidebar-content';
  content.style.cssText = `
    flex: 1;
    overflow-y: auto;
    padding: 20px;
    scroll-behavior: smooth;
  `;

  // 底部控制栏
  const controls = document.createElement('div');
  controls.style.cssText = `
    padding: 20px;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    background: rgba(15, 23, 42, 0.5);
    display: flex;
    justify-content: center;
    gap: 16px;
  `;

  const pauseBtn = document.createElement('button');
  pauseBtn.id = 'realtime-ai-pause-btn';
  pauseBtn.textContent = '⏸️ 暂停';
  pauseBtn.className = 'sidebar-btn';
  pauseBtn.onclick = togglePause;

  controls.appendChild(pauseBtn);

  container.appendChild(header);
  container.appendChild(content);
  container.appendChild(controls);

  document.body.appendChild(container);
  scriptState.sidebarContainer = container;

  // 注入样式
  const style = document.createElement('style');
  style.textContent = `
    .sidebar-sentence {
      padding: 12px;
      margin-bottom: 8px;
      border-radius: 8px;
      cursor: pointer;
      transition: all 0.2s;
      color: #cbd5e1;
      line-height: 1.6;
      font-size: 15px;
    }
    .sidebar-sentence:hover {
      background: rgba(255, 255, 255, 0.05);
    }
    .sidebar-sentence.active {
      background: rgba(99, 102, 241, 0.2);
      color: #fff;
      border-left: 3px solid #6366f1;
    }
    .sidebar-btn {
      background: #6366f1;
      color: white;
      border: none;
      padding: 8px 24px;
      border-radius: 8px;
      font-weight: 500;
      cursor: pointer;
      transition: all 0.2s;
    }
    .sidebar-btn:hover {
      background: #4f46e5;
    }
    #realtime-ai-sidebar-content::-webkit-scrollbar {
      width: 6px;
    }
    #realtime-ai-sidebar-content::-webkit-scrollbar-track {
      background: rgba(255, 255, 255, 0.05);
    }
    #realtime-ai-sidebar-content::-webkit-scrollbar-thumb {
      background: rgba(255, 255, 255, 0.2);
      border-radius: 3px;
    }
  `;
  document.head.appendChild(style);

  return container;
}

// 渲染侧边栏内容
function renderSidebarContent(sentences) {
  const content = document.getElementById('realtime-ai-sidebar-content');
  if (!content) return;

  content.innerHTML = '';
  sentences.forEach((text, index) => {
    const div = document.createElement('div');
    div.className = 'sidebar-sentence';
    div.textContent = text;
    div.dataset.index = index;
    div.onclick = () => jumpToSentence(index);
    content.appendChild(div);
  });
}

// 高亮当前句子
function highlightSidebarSentence(index) {
  const content = document.getElementById('realtime-ai-sidebar-content');
  if (!content) return;

  const prev = content.querySelector('.sidebar-sentence.active');
  if (prev) prev.classList.remove('active');

  const current = content.querySelector(`.sidebar-sentence[data-index="${index}"]`);
  if (current) {
    current.classList.add('active');
    // 滚动到可见区域
    current.scrollIntoView({ behavior: 'smooth', block: 'center' });
  }
}

// 切换暂停状态
function togglePause() {
  const btn = document.getElementById('realtime-ai-pause-btn');
  if (scriptState.readerState.isPaused) {
    scriptState.readerState.isPaused = false;
    btn.textContent = '⏸️ 暂停';
    playNextSentence(); // 恢复播放
  } else {
    scriptState.readerState.isPaused = true;
    btn.textContent = '▶️ 继续';
    // 停止当前音频
    if (scriptState.readerState.audioContext) {
      scriptState.readerState.audioContext.suspend();
    }
  }
}

// 跳转到指定句子
function jumpToSentence(index) {
  scriptState.readerState.currentIndex = index;
  scriptState.readerState.isPaused = false;
  const btn = document.getElementById('realtime-ai-pause-btn');
  if (btn) btn.textContent = '⏸️ 暂停';

  // 停止当前音频并重新开始
  if (scriptState.readerState.audioContext) {
    scriptState.readerState.audioContext.close();
    scriptState.readerState.audioContext = null;
  }
  playNextSentence();
}

// PCM 转 WAV
function pcmToWav(pcmData, sampleRate = 24000) {
  const numChannels = 1;
  const bitsPerSample = 16;
  const dataLength = pcmData.byteLength;
  const arrayBuffer = new ArrayBuffer(44 + dataLength);
  const view = new DataView(arrayBuffer);

  const writeString = (offset, string) => {
    for (let i = 0; i < string.length; i++) {
      view.setUint8(offset + i, string.charCodeAt(i));
    }
  };

  writeString(0, 'RIFF');
  view.setUint32(4, 36 + dataLength, true);
  writeString(8, 'WAVE');
  writeString(12, 'fmt ');
  view.setUint32(16, 16, true);
  view.setUint16(20, 1, true);
  view.setUint16(22, numChannels, true);
  view.setUint32(24, sampleRate, true);
  view.setUint32(28, sampleRate * numChannels * (bitsPerSample / 8), true);
  view.setUint16(32, numChannels * (bitsPerSample / 8), true);
  view.setUint16(34, bitsPerSample, true);
  writeString(36, 'data');
  view.setUint32(40, dataLength, true);

  new Uint8Array(arrayBuffer, 44).set(new Uint8Array(pcmData));

  return arrayBuffer;
}

// 播放音频
async function playAudio(base64Data) {
  return new Promise((resolve, reject) => {
    try {
      const binaryString = atob(base64Data);
      const len = binaryString.length;
      const bytes = new Uint8Array(len);
      for (let i = 0; i < len; i++) {
        bytes[i] = binaryString.charCodeAt(i);
      }

      const wavBuffer = pcmToWav(bytes.buffer);

      if (!scriptState.readerState.audioContext) {
        scriptState.readerState.audioContext = new (window.AudioContext || window.webkitAudioContext)();
      }

      // 确保 context 是 running 状态
      if (scriptState.readerState.audioContext.state === 'suspended') {
        scriptState.readerState.audioContext.resume();
      }

      scriptState.readerState.audioContext.decodeAudioData(wavBuffer, (buffer) => {
        const source = scriptState.readerState.audioContext.createBufferSource();
        source.buffer = buffer;
        source.connect(scriptState.readerState.audioContext.destination);

        source.onended = () => {
          resolve();
        };

        source.start(0);
      }, (error) => {
        reject(error);
      });
    } catch (error) {
      reject(error);
    }
  });
}

// 播放下一句
async function playNextSentence() {
  const state = scriptState.readerState;
  if (!state.isPlaying || state.isPaused) return;

  if (state.currentIndex >= state.sentences.length) {
    stopSidebarReader();
    return;
  }

  const sentence = state.sentences[state.currentIndex];
  highlightSidebarSentence(state.currentIndex);

  try {
    // 请求 TTS
    const response = await chrome.runtime.sendMessage({
      type: 'TTS_SYNTHESIZE',
      params: {
        text: sentence,
        voiceId: state.voiceId
      }
    });

    if (!response.success) {
      throw new Error(response.error);
    }

    const audioData = response.data.audio || response.data.data?.audio;
    if (audioData) {
      await playAudio(audioData);

      // 播放完成后，如果状态没变，继续下一句
      if (state.isPlaying && !state.isPaused) {
        state.currentIndex++;
        playNextSentence();
      }
    }
  } catch (error) {
    console.error('[Realtime AI] TTS Playback error:', error);
    showNotification('播放出错: ' + error.message, 'error');
    // 出错后暂停
    state.isPaused = true;
    const btn = document.getElementById('realtime-ai-pause-btn');
    if (btn) btn.textContent = '▶️ 重试';
  }
}

// 启动侧边栏阅读器
function startSidebarReader(sentences, voiceId) {
  const container = createSidebar();

  // 重置状态
  scriptState.readerState = {
    isPlaying: true,
    isPaused: false,
    sentences: sentences,
    currentIndex: 0,
    audioContext: null,
    voiceId: voiceId
  };

  renderSidebarContent(sentences);

  // 显示侧边栏
  requestAnimationFrame(() => {
    container.style.display = 'flex';
    requestAnimationFrame(() => {
      container.style.transform = 'translateX(0)';
    });
  });

  // 开始播放
  playNextSentence();
}

// 停止侧边栏阅读器
function stopSidebarReader() {
  scriptState.readerState.isPlaying = false;
  scriptState.readerState.isPaused = false;

  if (scriptState.readerState.audioContext) {
    scriptState.readerState.audioContext.close();
    scriptState.readerState.audioContext = null;
  }

  const container = scriptState.sidebarContainer;
  if (container) {
    container.style.transform = 'translateX(100%)';
    setTimeout(() => {
      container.style.display = 'none';
    }, 300);
  }
}

// 更新消息监听
const existingListener = chrome.runtime.onMessage.hasListeners();
// 注意：这里我们不能直接移除旧的 listener，因为它是匿名函数。
// 但我们可以添加新的 listener，只要它处理新的消息类型即可。

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'START_SIDEBAR_READER':
      startSidebarReader(message.sentences, message.voiceId);
      sendResponse({ success: true });
      break;

    case 'STOP_SIDEBAR_READER':
      stopSidebarReader();
      sendResponse({ success: true });
      break;
  }
  // 注意：不要返回 true，除非我们需要异步 sendResponse
  // 之前的 listener 已经处理了其他消息并返回 true
});

console.log('[Realtime AI] Content script loaded (v2)');
