// offscreen.js - Offscreen document for TRTC SDK handling

console.log('[Realtime AI] ======================================');
console.log('[Realtime AI] Offscreen JS loaded');
console.log('[Realtime AI] Location:', window.location.href);
console.log('[Realtime AI] TRTC defined:', typeof TRTC !== 'undefined');
console.log('[Realtime AI] ======================================');

let trtcClient = null;
let audioStream = null;

// 初始化 TRTC 客户端
function initTRTCClient() {
  if (trtcClient) {
    return; // 已经初始化
  }

  if (typeof TRTC === 'undefined') {
    throw new Error('TRTC SDK 未加载');
  }

  trtcClient = TRTC.create();

  // 监听自定义消息
  trtcClient.on(TRTC.EVENT.CUSTOM_MESSAGE, (event) => {
    try {
      // event.data 可能是 ArrayBuffer 或字符串
      let messageData = event.data;

      // 如果是 ArrayBuffer，先转换为字符串
      if (messageData instanceof ArrayBuffer) {
        const decoder = new TextDecoder('utf-8');
        messageData = decoder.decode(messageData);
      }

      // 解析 JSON
      const message = JSON.parse(messageData);

      console.log('[Realtime AI] 收到 TRTC 消息:', message);

      // 检查是否是翻译消息 (type=10000)
      if (message.type === 10000 && message.payload) {
        console.log('[Realtime AI] 翻译消息:', message.payload);

        // 转发消息到 background
        chrome.runtime.sendMessage({
          type: 'TRTC_TRANSLATION_MESSAGE',
          data: message.payload
        }).catch(() => {});
      }

    } catch (error) {
      console.error('[Realtime AI] 解析 TRTC 消息失败:', error);
    }
  });

  // 监听远程用户发布音频
  trtcClient.on(TRTC.EVENT.REMOTE_AUDIO_AVAILABLE, (event) => {
    console.log('[Realtime AI] Remote audio available:', event.userId);
  });

  // 监听远程用户取消发布音频
  trtcClient.on(TRTC.EVENT.REMOTE_AUDIO_UNAVAILABLE, (event) => {
    console.log('[Realtime AI] Remote audio unavailable:', event.userId);
  });

  // 监听错误
  trtcClient.on(TRTC.EVENT.ERROR, (error) => {
    console.error('[Realtime AI] TRTC 错误:', error);
    chrome.runtime.sendMessage({
      type: 'TRTC_ERROR',
      error: error.message
    }).catch(() => {});
  });

  console.log('[Realtime AI] TRTC Client initialized');
}

// 加入 TRTC 房间
async function joinRoom({ roomId, userId, userSig, sdkAppId }) {
  console.log('[Realtime AI] joinRoom called with:', { roomId, userId, sdkAppId });

  initTRTCClient();

  if (!trtcClient) {
    throw new Error('TRTC Client 初始化失败');
  }

  try {
    // 先尝试退出现有房间（如果有的话）
    try {
      await trtcClient.exitRoom();
      console.log('[Realtime AI] Exited previous room');
      // 等待退出完成
      await new Promise(resolve => setTimeout(resolve, 500));
    } catch (exitError) {
      // 如果没有在房间中，会报错，忽略
      console.log('[Realtime AI] No previous room to exit');
    }

    await trtcClient.enterRoom({
      roomId: roomId,
      sdkAppId: sdkAppId,
      userId: userId,
      userSig: userSig,
      scene: 'rtc'
    });

    console.log('[Realtime AI] Joined TRTC room:', roomId);
    return { success: true };
  } catch (error) {
    console.error('[Realtime AI] Failed to join room:', error);
    throw error;
  }
}

// 离开 TRTC 房间
async function leaveRoom() {
  if (trtcClient) {
    await trtcClient.exitRoom();
    console.log('[Realtime AI] Left TRTC room');
  }
  return { success: true };
}

// 开始标签页音频采集并推流
async function startTabCapture(streamId) {
  if (!trtcClient) {
    throw new Error('TRTC 客户端未初始化');
  }

  try {
    console.log('[Realtime AI] Starting tab capture with streamId:', streamId);

    // 使用 streamId 获取标签页音频流
    audioStream = await navigator.mediaDevices.getUserMedia({
      audio: {
        mandatory: {
          chromeMediaSource: 'tab',
          chromeMediaSourceId: streamId
        }
      },
      video: false
    });

    const audioTrack = audioStream.getAudioTracks()[0];
    console.log('[Realtime AI] Got audio track:', audioTrack.label);

    // 使用 TRTC 自定义采集推流
    await trtcClient.startLocalAudio({ option: { audioTrack } });

    console.log('[Realtime AI] Tab audio published to TRTC');
    return { success: true };

  } catch (error) {
    console.error('[Realtime AI] Failed to capture/publish tab audio:', error);
    throw error;
  }
}

// 发布音频流（保留兼容性）
async function publishAudio(streamId) {
  return startTabCapture(streamId);
}

// 停止发布音频
async function unpublishAudio() {
  if (trtcClient) {
    try {
      // 停止本地音频
      await trtcClient.stopLocalAudio();
      console.log('[Realtime AI] Stopped local audio');
    } catch (error) {
      console.error('[Realtime AI] Error stopping local audio:', error);
    }
  }

  if (audioStream) {
    // 停止所有音频轨道
    audioStream.getTracks().forEach(track => track.stop());
    audioStream = null;
    console.log('[Realtime AI] Audio stream stopped');
  }

  return { success: true };
}

// 监听来自 background 的消息
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // 只处理发送给 offscreen 的消息
  const offscreenMessages = ['TRTC_JOIN_ROOM', 'TRTC_LEAVE_ROOM', 'START_TAB_CAPTURE', 'STOP_TAB_CAPTURE', 'TRTC_PUBLISH_AUDIO', 'TRTC_UNPUBLISH_AUDIO', 'OFFSCREEN_READY'];

  if (!offscreenMessages.includes(message.type)) {
    // 不是发给 offscreen 的消息，忽略
    return false;
  }

  console.log('[Realtime AI] Offscreen received message:', message.type);

  (async () => {
    try {
      switch (message.type) {
        case 'TRTC_JOIN_ROOM':
          console.log('[Realtime AI] Offscreen joining room:', message.data);
          const joinResult = await joinRoom(message.data);
          sendResponse(joinResult);
          break;

        case 'TRTC_LEAVE_ROOM':
          const leaveResult = await leaveRoom();
          sendResponse(leaveResult);
          break;

        case 'START_TAB_CAPTURE':
          const captureResult = await startTabCapture(message.streamId);
          sendResponse(captureResult);
          break;

        case 'STOP_TAB_CAPTURE':
          // 停止音频捕获
          const stopResult = await unpublishAudio();
          sendResponse(stopResult);
          break;

        case 'TRTC_PUBLISH_AUDIO':
          const publishResult = await publishAudio(message.streamId);
          sendResponse(publishResult);
          break;

        case 'TRTC_UNPUBLISH_AUDIO':
          const unpublishResult = await unpublishAudio();
          sendResponse(unpublishResult);
          break;

        default:
          sendResponse({ success: false, error: '未知消息类型' });
      }
    } catch (error) {
      console.error('[Realtime AI] Offscreen error:', error);
      sendResponse({ success: false, error: error.message });
    }
  })();

  return true; // 保持消息通道开放
});

// 等待 TRTC SDK 加载完成
function waitForTRTC() {
  return new Promise((resolve, reject) => {
    if (typeof TRTC !== 'undefined') {
      resolve();
      return;
    }

    let attempts = 0;
    const maxAttempts = 50; // 最多等待 5 秒
    const checkInterval = setInterval(() => {
      attempts++;
      if (typeof TRTC !== 'undefined') {
        clearInterval(checkInterval);
        resolve();
      } else if (attempts >= maxAttempts) {
        clearInterval(checkInterval);
        reject(new Error('TRTC SDK 加载超时'));
      }
    }, 100);
  });
}

// 初始化检查
waitForTRTC()
  .then(() => {
    console.log('[Realtime AI] Offscreen document loaded, TRTC SDK ready');
    // 通知 background 已就绪
    console.log('[Realtime AI] Sending OFFSCREEN_READY message...');
    chrome.runtime.sendMessage({
      type: 'OFFSCREEN_READY'
    })
    .then((response) => {
      console.log('[Realtime AI] OFFSCREEN_READY message sent, response:', response);
    })
    .catch((error) => {
      console.error('[Realtime AI] Failed to send OFFSCREEN_READY:', error);
    });
  })
  .catch((error) => {
    console.error('[Realtime AI] Failed to load TRTC SDK:', error);
  });
