// popup.js - Popup UI logic for Chrome Realtime AI Extension

// ============================================================================
// UI 元素引用
// ============================================================================

// 登录界面元素
const loginScreen = document.getElementById('loginScreen');
const mainScreen = document.getElementById('mainScreen');
const emailInput = document.getElementById('emailInput');
const otpInput = document.getElementById('otpInput');
const sendOtpBtn = document.getElementById('sendOtpBtn');
const verifyOtpBtn = document.getElementById('verifyOtpBtn');
const otpSection = document.getElementById('otpSection');
const errorMsg = document.getElementById('errorMsg');
const mainErrorMsg = document.getElementById('mainErrorMsg');

// 主界面元素
const menuBtn = document.getElementById('menuBtn');
const menuDropdown = document.getElementById('menuDropdown');
const logoutMenuItem = document.getElementById('logoutMenuItem');
const quotaValue = document.getElementById('quotaValue');
const quotaPlan = document.getElementById('quotaPlan');

// 翻译控制元素
const sourceLang = document.getElementById('sourceLang');
const targetLang = document.getElementById('targetLang');
const swapLangBtn = document.getElementById('swapLangBtn');
const translationMode = document.getElementById('translationMode');
const translatorVoiceSelect = document.getElementById('translatorVoiceSelect');
const translatorVoiceSettings = document.getElementById('translatorVoiceSettings');
const startBtn = document.getElementById('startBtn');
const stopBtn = document.getElementById('stopBtn');
const statusSection = document.getElementById('statusSection');
const statusText = document.getElementById('statusText');
const volumeLevel = document.getElementById('volumeLevel');

// 阅读器元素
const voiceSelect = document.getElementById('voiceSelect');
const readerStartBtn = document.getElementById('readerStartBtn');
const readerPauseBtn = document.getElementById('readerPauseBtn');
const readerStopBtn = document.getElementById('readerStopBtn');
const currentTextSection = document.getElementById('currentTextSection');
const currentText = document.getElementById('currentText');
const progressText = document.getElementById('progressText');
const readerErrorMsg = document.getElementById('readerErrorMsg');

// ============================================================================
// 应用状态
// ============================================================================

let currentEmail = '';
let isRunning = false;

// 阅读器状态
let readerState = {
  isPlaying: false,
  isPaused: false,
  sentences: [],
  currentIndex: 0,
  audioContext: null,
  currentSource: null
};

// ============================================================================
// 通用辅助函数
// ============================================================================

/**
 * 切换登录/主界面显示
 */
function showScreen(screen) {
  if (screen === 'login') {
    closeMenu();
    loginScreen.classList.remove('hidden');
    mainScreen.classList.add('hidden');
  } else {
    loginScreen.classList.add('hidden');
    mainScreen.classList.remove('hidden');
  }
}

/**
 * 显示错误消息（5秒后自动隐藏）
 */
function showError(element, message) {
  element.textContent = message;
  element.classList.remove('hidden');
  setTimeout(() => {
    element.classList.add('hidden');
  }, 5000);
}

/**
 * 获取当前活跃的标签页
 */
async function getActiveTab() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  return tab;
}

/**
 * 菜单控制
 */
function closeMenu() {
  if (menuDropdown) {
    menuDropdown.classList.add('hidden');
  }
  if (menuBtn) {
    menuBtn.setAttribute('aria-expanded', 'false');
  }
}

function toggleMenu(event) {
  event.stopPropagation();
  if (!menuDropdown || !menuBtn) return;

  const willOpen = menuDropdown.classList.contains('hidden');
  if (willOpen) {
    menuDropdown.classList.remove('hidden');
    menuBtn.setAttribute('aria-expanded', 'true');
  } else {
    closeMenu();
  }
}

function setupMenu() {
  if (!menuBtn || !menuDropdown) return;

  menuBtn.addEventListener('click', toggleMenu);
  logoutMenuItem?.addEventListener('click', (event) => {
    event.stopPropagation();
    closeMenu();
    handleLogout();
  });

  document.addEventListener('click', (event) => {
    if (menuDropdown.contains(event.target) || menuBtn.contains(event.target)) {
      return;
    }
    closeMenu();
  });
}

/**
 * 解析配额响应数据
 */
function parseQuotaResponse(response) {
  const quotaData = response.data?.quota || response.data;
  return {
    remaining: quotaData?.remaining ?? 0,
    dailyLimit: quotaData?.daily ?? 100,
    tier: quotaData?.subscription_tier ?? 'free'
  };
}

/**
 * 更新配额显示
 */
function updateQuotaDisplay(remaining, dailyLimit, tier) {
  // 显示为 "977/1000" 格式
  quotaValue.textContent = `${remaining}/${dailyLimit}`;

  // 将 subscription_tier 转换为显示文本
  const tierMap = {
    'free': 'Free',
    'pro': 'Pro',
    'max': 'Max',
    'premium': 'Premium'
  };
  quotaPlan.textContent = tierMap[tier?.toLowerCase()] || 'Free';
}

// ============================================================================
// 登录认证功能
// ============================================================================

/**
 * 发送 OTP 验证码
 */
async function handleSendOtp() {
  const email = emailInput.value.trim();

  if (!email || !email.includes('@')) {
    showError(errorMsg, '请输入有效的邮箱地址');
    return;
  }

  currentEmail = email;
  sendOtpBtn.disabled = true;
  sendOtpBtn.textContent = '发送中...';

  try {
    const response = await chrome.runtime.sendMessage({
      type: 'LOGIN',
      email: email
    });

    if (response.success) {
      otpSection.classList.remove('hidden');
      sendOtpBtn.textContent = '重新发送';
      otpInput.focus();
    } else {
      throw new Error(response.error || '发送失败');
    }
  } catch (error) {
    showError(errorMsg, error.message);
    sendOtpBtn.textContent = '发送验证码';
  } finally {
    sendOtpBtn.disabled = false;
  }
}

/**
 * 验证 OTP 并登录
 */
async function handleVerifyOtp() {
  const otp = otpInput.value.trim();

  if (!otp || otp.length !== 6) {
    showError(errorMsg, '请输入6位验证码');
    return;
  }

  verifyOtpBtn.disabled = true;
  verifyOtpBtn.textContent = '验证中...';

  try {
    const response = await chrome.runtime.sendMessage({
      type: 'VERIFY_OTP',
      email: currentEmail,
      otp: otp
    });

    if (response.success) {
      await loadMainScreen();
    } else {
      throw new Error(response.error || '验证失败');
    }
  } catch (error) {
    showError(errorMsg, error.message);
    verifyOtpBtn.textContent = '验证登录';
    verifyOtpBtn.disabled = false;
  }
}

/**
 * 退出登录
 */
async function handleLogout() {
  if (!confirm('确定要退出登录吗?')) {
    return;
  }

  try {
    await chrome.runtime.sendMessage({ type: 'LOGOUT' });
    showScreen('login');
    emailInput.value = '';
    otpInput.value = '';
    otpSection.classList.add('hidden');
    currentEmail = '';
  } catch (error) {
    showError(mainErrorMsg, error.message);
  }
}

// ============================================================================
// 配额管理
// ============================================================================

/**
 * 加载主界面并初始化配额
 */
async function loadMainScreen() {
  showScreen('main');

  try {
    await loadQuota();

    /**
     * 检查是否有活动的翻译 session
     * 如果用户在其他标签页打开 popup，需要恢复 UI 状态
     * 这样才能显示"停止翻译"按钮
     */
    const response = await chrome.runtime.sendMessage({ type: 'GET_SESSION' });
    if (response.success && response.session) {
      // 有活动的 session，恢复 UI 状态
      isRunning = true;
      startBtn.classList.add('hidden');
      stopBtn.classList.remove('hidden');
      statusSection.classList.remove('hidden');
      statusText.textContent = '翻译中...';
      console.log('[Popup] Restored active session UI');
    }
  } catch (error) {
    console.error('[Popup] Failed to load quota on init:', error);
  }
}

/**
 * 获取并显示配额信息
 */
async function loadQuota() {
  try {
    const response = await chrome.runtime.sendMessage({ type: 'GET_QUOTA' });

    if (response.success && response.data) {
      const { remaining, dailyLimit, tier } = parseQuotaResponse(response);
      updateQuotaDisplay(remaining, dailyLimit, tier);
    } else {
      throw new Error(response?.error || '获取配额失败');
    }
  } catch (error) {
    console.error('[Popup] Load quota error:', error);

    // 检查是否是登录过期错误
    if (error.message?.includes('登录已过期') || error.message?.includes('未登录')) {
      await chrome.storage.local.remove('authSession');
      showScreen('login');
      showError(errorMsg, '登录已过期，请重新登录');
      return;
    }

    showError(mainErrorMsg, error.message || '获取配额失败');

    // 显示默认值
    quotaValue.textContent = '--/--';
    quotaPlan.textContent = 'Max';
  }
}

// ============================================================================
// 翻译功能
// ============================================================================

/**
 * 互换源语言和目标语言
 */
function swapLanguages() {
  const temp = sourceLang.value;
  sourceLang.value = targetLang.value;
  targetLang.value = temp;
}

/**
 * 开始翻译
 */
async function handleStart() {
  startBtn.disabled = true;
  startBtn.textContent = '启动中...';

  try {
    const params = {
      sourceLang: sourceLang.value,
      targetLang: targetLang.value,
      mode: translationMode.value
    };

    // 如果是语音模式，添加音色 ID
    if (translationMode.value === 'voice') {
      params.voiceId = await getSelectedVoice();
    }

    const response = await chrome.runtime.sendMessage({
      type: 'START_INTERPRETATION',
      params: params
    });

    if (response.success) {
      isRunning = true;
      startBtn.classList.add('hidden');
      stopBtn.classList.remove('hidden');
      statusSection.classList.remove('hidden');
      statusText.textContent = '翻译中...';

      await loadQuota();
    } else {
      throw new Error(response.error || '启动失败');
    }
  } catch (error) {
    showError(mainErrorMsg, error.message);
    startBtn.textContent = '▶️ 开始翻译';
  } finally {
    startBtn.disabled = false;
  }
}

/**
 * 停止翻译
 */
async function handleStop() {
  stopBtn.disabled = true;
  stopBtn.textContent = '停止中...';

  try {
    const response = await chrome.runtime.sendMessage({
      type: 'STOP_INTERPRETATION'
    });

    if (response.success) {
      isRunning = false;
      stopBtn.classList.add('hidden');
      startBtn.classList.remove('hidden');
      statusSection.classList.add('hidden');
      startBtn.textContent = '▶️ 开始翻译';

      await loadQuota();
    } else {
      throw new Error(response.error || '停止失败');
    }
  } catch (error) {
    showError(mainErrorMsg, error.message);
    stopBtn.textContent = '⏹️ 停止翻译';
  } finally {
    stopBtn.disabled = false;
  }
}

/**
 * 处理 TRTC 实时消息
 */
function handleTRTCMessage(message) {
  switch (message.type) {
    case 'conversation':
      if (message.data.volume !== undefined) {
        const volumePercent = Math.min(message.data.volume * 100, 100);
        volumeLevel.style.width = `${volumePercent}%`;
      }
      break;

    case 'state_change':
      statusText.textContent = message.data.state || '翻译中...';
      break;

    case 'error':
      showError(mainErrorMsg, message.data.message || '发生错误');
      break;
  }
}

// ============================================================================
// TTS 阅读器功能
// ============================================================================

/**
 * 初始化标签页切换
 */
function initTabSwitching() {
  const tabBtns = document.querySelectorAll('.tab-btn');
  const translatorTab = document.getElementById('translatorTab');
  const readerTab = document.getElementById('readerTab');

  if (!tabBtns.length || !translatorTab || !readerTab) {
    console.error('[Popup] Tab elements not found');
    return;
  }

  tabBtns.forEach(btn => {
    btn.addEventListener('click', () => {
      const tabName = btn.dataset.tab;

      tabBtns.forEach(b => b.classList.remove('active'));
      btn.classList.add('active');

      if (tabName === 'translator') {
        translatorTab.classList.remove('hidden');
        readerTab.classList.add('hidden');
      } else if (tabName === 'reader') {
        translatorTab.classList.add('hidden');
        readerTab.classList.remove('hidden');
      }
    });
  });
}

/**
 * 初始化阅读器事件
 */
function initReaderEvents() {
  if (!readerStartBtn || !readerPauseBtn || !readerStopBtn) {
    console.error('[Popup] Reader button elements not found');
    return;
  }

  readerStartBtn.addEventListener('click', handleReaderStart);
  readerPauseBtn.addEventListener('click', handleReaderPause);
  readerStopBtn.addEventListener('click', handleReaderStop);
}

/**
 * 获取可用的 TTS 音色列表
 */
async function fetchVoiceList() {
  try {
    const response = await fetch('https://api.realtime-ai.chat/api/tts/voices', {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json'
      }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch voices: ${response.status}`);
    }

    const data = await response.json();
    return data.voices || [];
  } catch (error) {
    console.error('[Popup] Failed to fetch voices:', error);
    // 返回默认音色
    return [{
      id: 'v-female-R2s4N9qJ',
      name: '小芮',
      language: 'zh',
      description: '默认女声'
    }];
  }
}

/**
 * 获取已选择的音色 ID
 */
async function getSelectedVoice() {
  const result = await chrome.storage.local.get('selectedVoiceId');
  return result.selectedVoiceId || 'v-female-R2s4N9qJ';
}

/**
 * 保存选择的音色 ID
 */
async function saveSelectedVoice(voiceId) {
  await chrome.storage.local.set({ selectedVoiceId: voiceId });
}

/**
 * 初始化音色选择器（阅读器）
 */
async function initVoiceSelector() {
  if (!voiceSelect) {
    console.error('[Popup] Voice select element not found');
    return;
  }

  // 获取音色列表
  const voices = await fetchVoiceList();

  // 清空加载选项
  voiceSelect.innerHTML = '';

  // 填充下拉框
  voices.forEach(voice => {
    const option = document.createElement('option');
    option.value = voice.id;
    option.textContent = `${voice.name} (${voice.language || 'zh'})`;
    if (voice.description) {
      option.title = voice.description;
    }
    voiceSelect.appendChild(option);
  });

  // 加载已保存的选择
  const savedVoiceId = await getSelectedVoice();
  voiceSelect.value = savedVoiceId;

  // 监听选择变化
  voiceSelect.addEventListener('change', async () => {
    await saveSelectedVoice(voiceSelect.value);
  });
}

/**
 * 初始化翻译器音色选择器
 */
async function initTranslatorVoiceSelector() {
  if (!translatorVoiceSelect) {
    console.error('[Popup] Translator voice select element not found');
    return;
  }

  // 获取音色列表（复用已有函数）
  const voices = await fetchVoiceList();

  // 清空加载选项
  translatorVoiceSelect.innerHTML = '';

  // 填充下拉框
  voices.forEach(voice => {
    const option = document.createElement('option');
    option.value = voice.id;
    option.textContent = `${voice.name} (${voice.language || 'zh'})`;
    if (voice.description) {
      option.title = voice.description;
    }
    translatorVoiceSelect.appendChild(option);
  });

  // 加载已保存的选择（与阅读器共享同一个存储键）
  const savedVoiceId = await getSelectedVoice();
  translatorVoiceSelect.value = savedVoiceId;

  // 监听选择变化
  translatorVoiceSelect.addEventListener('change', async () => {
    await saveSelectedVoice(translatorVoiceSelect.value);
    // 同步更新阅读器的选择器（如果存在）
    if (voiceSelect) {
      voiceSelect.value = translatorVoiceSelect.value;
    }
  });
}

/**
 * 根据翻译模式显示/隐藏音色选择器
 */
function updateTranslatorVoiceVisibility() {
  if (!translatorVoiceSettings || !translationMode) return;

  if (translationMode.value === 'voice') {
    translatorVoiceSettings.classList.remove('hidden');
  } else {
    translatorVoiceSettings.classList.add('hidden');
  }
}

/**
 * 文本分句
 */
function splitIntoSentences(text) {
  const sentences = text.match(/[^.!?。!?]+[.!?。!?]+|[^.!?。!?]+$/g) || [];
  return sentences
    .map(s => s.trim())
    .filter(s => s.length > 0 && s.length < 150);
}

/**
 * 开始阅读
 */
async function handleReaderStart() {
  try {
    readerStartBtn.disabled = true;
    readerStartBtn.textContent = '提取中...';

    const tab = await getActiveTab();
    if (!tab) {
      throw new Error('无法获取当前标签页');
    }

    // 提取文本
    let response;
    try {
      response = await chrome.tabs.sendMessage(tab.id, { type: 'EXTRACT_TEXT' });
    } catch (error) {
      // 动态注入 content script
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ['content.js']
      });

      await new Promise(resolve => setTimeout(resolve, 500));
      response = await chrome.tabs.sendMessage(tab.id, { type: 'EXTRACT_TEXT' });
    }

    if (!response || !response.success || !response.text) {
      throw new Error('无法提取页面文本');
    }

    // 分句
    const sentences = splitIntoSentences(response.text);
    if (sentences.length === 0) {
      throw new Error('页面没有可读取的文本');
    }

    // 获取选择的音色
    const selectedVoiceId = await getSelectedVoice();

    // 检查是否启用侧边栏模式
    const sidebarToggle = document.getElementById('sidebarModeToggle');
    const useSidebar = sidebarToggle ? sidebarToggle.checked : true;

    if (useSidebar) {
      // 启动侧边栏阅读器
      await chrome.tabs.sendMessage(tab.id, {
        type: 'START_SIDEBAR_READER',
        sentences: sentences,
        voiceId: selectedVoiceId
      });

      // 关闭 popup，让用户专注页面
      window.close();
    } else {
      // 使用 Popup 阅读器
      readerState.sentences = sentences;
      readerState.isPlaying = true;
      readerState.isPaused = false;
      readerState.currentIndex = 0;

      // 初始化 AudioContext
      if (!readerState.audioContext) {
        readerState.audioContext = new (window.AudioContext || window.webkitAudioContext)();
      }

      // 更新 UI
      readerStartBtn.classList.add('hidden');
      readerPauseBtn.classList.remove('hidden');
      readerStopBtn.classList.remove('hidden');
      currentTextSection.classList.remove('hidden');

      // 开始播放
      await playNextSentence();
    }

  } catch (error) {
    console.error('[Popup] Reader start failed:', error);
    showError(readerErrorMsg, error.message);
    readerStartBtn.textContent = '▶️ 开始阅读';
    readerStartBtn.disabled = false;
  }
}

/**
 * 播放下一句
 */
async function playNextSentence() {
  if (!readerState.isPlaying || readerState.isPaused) {
    return;
  }

  if (readerState.currentIndex >= readerState.sentences.length) {
    handleReaderStop();
    return;
  }

  const sentence = readerState.sentences[readerState.currentIndex];
  currentText.textContent = sentence;
  progressText.textContent = `${readerState.currentIndex + 1} / ${readerState.sentences.length}`;

  try {
    // 获取认证信息
    const { authSession } = await chrome.storage.local.get('authSession');
    if (!authSession || !authSession.access_token) {
      throw new Error('未登录，请先登录');
    }

    // 获取选择的音色
    const selectedVoiceId = await getSelectedVoice();

    // 调用 TTS API
    const response = await fetch('https://api.realtime-ai.chat/api/tts/synthesize', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authSession.access_token}`
      },
      body: JSON.stringify({
        text: sentence,
        voiceId: selectedVoiceId
      })
    });

    if (!response.ok) {
      if (response.status === 401) {
        throw new Error('登录已过期，请重新登录');
      }
      const errorText = await response.text();
      throw new Error(`TTS API 错误: ${response.status} - ${errorText}`);
    }

    const data = await response.json();

    // 获取音频数据
    const success = data.code === 'success' || data.success === true;
    const audioData = data.audio || data.data?.audio;

    if (!success || !audioData || typeof audioData !== 'string' || audioData.length === 0) {
      throw new Error(data.message || data.error || 'TTS 合成失败');
    }

    // 更新配额显示
    if (data.quota || data.data?.quota) {
      const { remaining, dailyLimit, tier } = parseQuotaResponse({ data });
      updateQuotaDisplay(remaining, dailyLimit, tier);
    }

    // 播放音频
    await playAudio(audioData);

    // 播放下一句
    readerState.currentIndex++;
    await playNextSentence();

  } catch (error) {
    console.error('[Popup] Playback failed:', error);
    showError(readerErrorMsg, error.message);
    handleReaderStop();
  }
}

/**
 * 将 PCM 转换为 WAV 格式
 */
function pcmToWav(pcmData, sampleRate = 24000) {
  const numChannels = 1;
  const bitsPerSample = 16;
  const dataLength = pcmData.byteLength;
  const arrayBuffer = new ArrayBuffer(44 + dataLength);
  const view = new DataView(arrayBuffer);

  const writeString = (offset, string) => {
    for (let i = 0; i < string.length; i++) {
      view.setUint8(offset + i, string.charCodeAt(i));
    }
  };

  writeString(0, 'RIFF');
  view.setUint32(4, 36 + dataLength, true);
  writeString(8, 'WAVE');
  writeString(12, 'fmt ');
  view.setUint32(16, 16, true);
  view.setUint16(20, 1, true);
  view.setUint16(22, numChannels, true);
  view.setUint32(24, sampleRate, true);
  view.setUint32(28, sampleRate * numChannels * bitsPerSample / 8, true);
  view.setUint16(32, numChannels * bitsPerSample / 8, true);
  view.setUint16(34, bitsPerSample, true);
  writeString(36, 'data');
  view.setUint32(40, dataLength, true);

  const pcmView = new Uint8Array(arrayBuffer, 44);
  pcmView.set(new Uint8Array(pcmData));

  return arrayBuffer;
}

/**
 * 播放 Base64 编码的 PCM 音频（使用 Web Audio API）
 */
function playAudio(base64Audio) {
  return new Promise((resolve, reject) => {
    try {
      if (!base64Audio || base64Audio.length === 0) {
        throw new Error('音频数据为空');
      }

      // Base64 解码
      const binaryString = atob(base64Audio);
      if (binaryString.length === 0) {
        throw new Error('Base64 解码后数据为空');
      }

      // 转换为 Uint8Array
      const bytes = new Uint8Array(binaryString.length);
      for (let i = 0; i < binaryString.length; i++) {
        bytes[i] = binaryString.charCodeAt(i);
      }

      // 转换 PCM 为 WAV
      const wavArrayBuffer = pcmToWav(bytes.buffer, 24000);

      // 使用 Web Audio API 解码并播放
      readerState.audioContext.decodeAudioData(wavArrayBuffer, (audioBuffer) => {
        // 停止之前的音频
        if (readerState.currentSource) {
          try {
            readerState.currentSource.stop();
          } catch (e) {
            // 忽略停止错误
          }
        }

        // 创建新的音频源
        const source = readerState.audioContext.createBufferSource();
        source.buffer = audioBuffer;
        source.connect(readerState.audioContext.destination);

        // 播放结束处理
        source.onended = () => {
          readerState.currentSource = null;
          resolve();
        };

        // 保存引用并开始播放
        readerState.currentSource = source;
        source.start(0);

      }, (error) => {
        console.error('[Popup] Audio decode error:', error);
        reject(new Error('音频解码失败'));
      });

    } catch (error) {
      console.error('[Popup] playAudio error:', error);
      reject(error);
    }
  });
}

/**
 * 暂停/继续阅读
 */
async function handleReaderPause() {
  if (readerState.isPaused) {
    // 继续播放
    readerState.isPaused = false;
    readerPauseBtn.textContent = '⏸️ 暂停';

    if (readerState.audioContext.state === 'suspended') {
      await readerState.audioContext.resume();
    } else {
      await playNextSentence();
    }
  } else {
    // 暂停
    readerState.isPaused = true;
    readerPauseBtn.textContent = '▶️ 继续';

    if (readerState.audioContext && readerState.audioContext.state === 'running') {
      await readerState.audioContext.suspend();
    }
  }
}

/**
 * 停止阅读
 */
function handleReaderStop() {
  // 停止当前音频
  if (readerState.currentSource) {
    try {
      readerState.currentSource.stop();
    } catch (error) {
      // 忽略停止错误
    }
  }
  readerState.currentSource = null;

  // 重置状态
  readerState.isPlaying = false;
  readerState.isPaused = false;
  readerState.currentIndex = 0;
  readerState.sentences = [];

  // 更新 UI
  readerStartBtn.classList.remove('hidden');
  readerPauseBtn.classList.add('hidden');
  readerStopBtn.classList.add('hidden');
  currentTextSection.classList.add('hidden');

  readerStartBtn.textContent = '▶️ 开始阅读';
  readerStartBtn.disabled = false;
  readerPauseBtn.textContent = '⏸️ 暂停';
}

// ============================================================================
// 应用初始化
// ============================================================================

/**
 * 显示扩展版本号
 */
function displayVersion() {
  const manifest = chrome.runtime.getManifest();
  const versionElement = document.getElementById('versionDisplay');
  if (versionElement && manifest.version) {
    versionElement.textContent = `v${manifest.version}`;
  }
}

/**
 * 主应用初始化
 */
async function init() {
  // 显示版本号
  displayVersion();
  // 检查登录状态
  const { authSession } = await chrome.storage.local.get('authSession');
  if (authSession && authSession.access_token) {
    await loadMainScreen();
  } else {
    showScreen('login');
  }

  // 绑定事件
  sendOtpBtn.addEventListener('click', handleSendOtp);
  verifyOtpBtn.addEventListener('click', handleVerifyOtp);
  setupMenu();
  startBtn.addEventListener('click', handleStart);
  stopBtn.addEventListener('click', handleStop);
  swapLangBtn?.addEventListener('click', swapLanguages);
  translationMode?.addEventListener('change', updateTranslatorVoiceVisibility);

  // 监听来自 background 的消息
  chrome.runtime.onMessage.addListener((message) => {
    if (message.type === 'TRTC_MESSAGE') {
      handleTRTCMessage(message.data);
    } else if (message.type === 'TRTC_ERROR') {
      showError(mainErrorMsg, message.error);
    }
  });
}

/**
 * DOM 加载完成后启动应用
 */
document.addEventListener('DOMContentLoaded', () => {
  try {
    initTabSwitching();
    initReaderEvents();
    initVoiceSelector();
    initTranslatorVoiceSelector();
    updateTranslatorVoiceVisibility();
    init();
  } catch (error) {
    console.error('[Popup] Initialization failed:', error);
  }
});
